from django.core.exceptions import ValidationError
from rest_framework.filters import BaseFilterBackend
from djangorestframework_camel_case.util import camel_to_underscore

class SimpleOrderingBackend(
    BaseFilterBackend,
):
    """
    Custom filter backend for sorting query sets.

    Requires a view set to define:
    - ordering_fields: list of allowed field names for sorting (default: [])
    - ordering_column_param: query parameter name for sort column (default: 'sort_by')
    - ordering_direction_param: query parameter name for a sort direction (default: 'sort_dir')
    """

    default_fields = [
        ("created", "created"),
        ("created_at", "created"),
        ("modified", "updated"),
        ("modified_at", "updated"),
        ("updated", "updated"),
        ("updated_at", "updated"),
    ]

    def get_allowed_fields(
        self,
        view
    ):
        allowed_fields = getattr(view, 'ordering_fields', []) or []

        return [
            *self.default_fields,
            *allowed_fields
        ]

    # noinspection PyMethodMayBeStatic
    def get_field(
        self,
        field_name_from_request: str,
        view,
    ) -> tuple[str, str]:
        """
        Return the field name to order by from the request.

        Returns:
            (field_name, database_field_name)
        """
        field_name_from_request = camel_to_underscore(
            field_name_from_request
        )

        allowed_fields = self.get_allowed_fields(view)

        for item in allowed_fields:
            if isinstance(item, str):
                if item == field_name_from_request:
                    return item, item
            elif isinstance(item, tuple) and len(item) == 2:
                request_name, mapped_field = item
                if request_name == field_name_from_request:
                    return request_name, mapped_field

        raise ValidationError(
            f'Invalid ordering field: {field_name_from_request}.',
        )

    # noinspection PyMethodMayBeStatic
    def get_ordering_params(
        self,
        request,
        view,
    ) -> tuple[str, str] | None:
        """Extract and validate ordering parameters from the request."""
        column = request.query_params.get(
            getattr(
                view,
                'ordering_column_param',
                'sort_by',
            ),
        )
        direction = request.query_params.get(
            getattr(
                view,
                'ordering_direction_param',
                'sort_dir',
            ),
        )

        if not column:
            return None

        if direction and direction.lower() not in ['asc', 'desc']:
            raise ValidationError(
                'Direction must be either "asc" or "desc"',
            )

        _, database_field_name = self.get_field(
            field_name_from_request=column,
            view=view,
        )

        return database_field_name, direction

    def filter_queryset(
        self,
        request,
        queryset,
        view,
    ):
        """Apply ordering to the queryset based on request parameters."""
        ordering_params = self.get_ordering_params(
            request,
            view,
        )

        if not ordering_params:
            return queryset

        column, direction = ordering_params
        if direction and direction.lower() == 'desc':
            column = f'-{column}'

        return queryset.order_by(
            column,
        )
