from django.db.models import Model
from django.db.models import CharField
from django.db.models import DateTimeField
from django.db.models import ForeignKey
from django.db.models import CASCADE
from django.db.models import FileField
from django.db.models import TextChoices
from django.db.models import IntegerField
from django.db.models import ImageField
from django.db.models import UniqueConstraint

from django.utils.translation import gettext_lazy as _

from userguide.managers.article import ArticleManager
from userguide.managers.category import CategoryManager
from userguide.managers.category_group import CategoryGroupManager

from mdeditor.fields import MDTextField


class BaseModel(Model):
    """
    Base model.

    Attributes:
        created (DateTimeField): "Created at" date time.
        updated (DateTimeField): "Updated at" time.
    """
    created = DateTimeField(auto_now_add=True, verbose_name=_('Created at'))
    updated = DateTimeField(auto_now=True, verbose_name=_('Updated at'))

    def get_pk(self):
        return self.id

    class Meta:
        ordering = ['-created']
        abstract = True

class CategoryGroup(BaseModel):
    """
    category group model.

    Attributes:
        description (str): Category group description, max length is 255.
        icon (str): `iconify/react` icon name to use at the frontend.
    """
    title = CharField(max_length=32, verbose_name=_('title'))
    order = IntegerField(default=0, verbose_name=_('Order'))
    description = CharField(max_length=255, verbose_name=_('Description'), blank=True, null=True)
    icon = CharField(max_length=64, blank=True, null=True, verbose_name=_('Icon'))

    objects = CategoryGroupManager()

    def __str__(self):
        return self.title

    class Meta:
        db_table = 'category_group'
        ordering = ['order']
        verbose_name = _('Category Group')
        verbose_name_plural = _('Category Groups')


class Category(BaseModel):
    """
    category model.

    Attributes:
        title (str): Category title.
        content (str): Category description, markdown.
    """
    title = CharField(max_length=32, verbose_name=_('Title'))
    content = MDTextField(verbose_name=_('Content'))
    order = IntegerField(default=0, verbose_name=_('Order'))

    image = ImageField(
        upload_to='user-guide/category-images/%Y/%m/%d',
        null=True,
        blank=True,
        verbose_name=_('Image'),
    )

    group = ForeignKey(
        to='CategoryGroup',
        on_delete=CASCADE,
        null=True,
        related_name='categories',
        verbose_name=_('Category group'),
    )

    objects = CategoryManager()

    def __str__(self):
        return self.title

    class Meta:
        db_table = 'category'
        ordering = ['order']
        verbose_name = _('Category')
        verbose_name_plural = _('Categories')

class ArticleAttachment(BaseModel):
    """
    article attachment
    """
    class Type(TextChoices):
        IMAGE = 'image', _('Image')
        VIDEO = 'video', _('Video')
        FILE = 'file', _('File')

    article = ForeignKey(to='Article', on_delete=CASCADE, related_name='attachments', verbose_name=_('Article'))
    file = FileField(upload_to='user-guide/article-attachments/%Y/%m/%d', verbose_name=_('File'))
    order = IntegerField(default=0, verbose_name=_('Order'))
    type = CharField(max_length=10, choices=Type.choices, verbose_name=_('Attachment type'))

    class Meta:
        db_table = 'article_attachment'
        ordering = ['order']
        verbose_name = _('Article Attachment')
        verbose_name_plural = _('Article Attachments')

class Article(BaseModel):
    """
    article

    Attributes:
        order (int):
            Order of the article.
            To sort articles from the highest priority to the lowest priority, use the order attribute.
    """
    category = ForeignKey(to='Category', on_delete=CASCADE, null=True, related_name='articles', verbose_name=_('Category'))
    title = CharField(max_length=255, verbose_name=_('Title'))
    content = MDTextField(verbose_name=_('Content'))

    order = IntegerField(default=0, verbose_name=_('Order'))

    objects = ArticleManager()

    def __str__(self):
        return self.title

    class Meta:
        db_table = 'article'
        ordering = ['order']
        verbose_name = _('Article')
        verbose_name_plural = _('Articles')


        constraints = [
            UniqueConstraint(
                fields=['category', 'order'],
                name='unique_category_order'
            )
        ]