from django.db.models import QuerySet
from django.db.models import Count
from django.db.models import Subquery
from django.db.models import OuterRef

from typing import Self

from django.apps import apps

class CategoryQuerySet(
    QuerySet,
):
    """
    Category QuerySet
    """

    def with_total_articles(
        self,
    ) -> Self:
        """
        Add total_articles field to queryset
        """
        return self.annotate(
            total_articles=Count(
                'articles',
            ),
        )

    def with_first_article_id(
        self,
    ) -> Self:
        """
        Add first_article_id field to queryset
        """
        # noinspection PyPep8Naming
        Article = apps.get_model('userguide', 'Article')

        # noinspection PyUnresolvedReferences
        article_query = Article.objects.filter(
            category=OuterRef(
                'pk',
            ),
        )

        return self.annotate(
            first_article_id=Subquery(
                article_query.order_by(
                    'order',
                ).values(
                    'id',
                )[:1],
            ),
        )

    def with_annotations(
        self,
    ) -> Self:
        """
        Applies with_total_articles method to queryset
        :return:
        """
        return self \
            .with_total_articles() \
            .with_first_article_id()
