# python
from django.contrib import admin
from django.db.models import Count
from django.utils.translation import gettext_lazy as _

from .models import Category, Article, ArticleAttachment, CategoryGroup


class ArticleAttachmentInline(admin.TabularInline):
    model = ArticleAttachment
    fields = ("file", "type", "created", "updated")
    readonly_fields = ("created", "updated")
    extra = 1
    show_change_link = True


@admin.register(Article)
class ArticleAdmin(admin.ModelAdmin):
    list_display = ("title", "category", "order", "attachments_count", "created", "updated")
    list_filter = ("category",)
    search_fields = ("title", "content", "category__title")
    readonly_fields = ("created", "updated")
    autocomplete_fields = ("category",)
    inlines = (ArticleAttachmentInline,)
    list_select_related = ("category",)
    ordering = ("-order", "title")
    date_hierarchy = "created"
    save_on_top = True
    list_per_page = 50

    fieldsets = (
        (_("Basics"), {"fields": ("title", "category", "order")}),
        (_("Content"), {"fields": ("content",)}),
        (_("Metadata"), {"fields": ("created", "updated"), "classes": ("collapse",)}),
    )

    def get_queryset(self, request):
        qs = super().get_queryset(request)
        return qs.select_related("category").annotate(_attachments_count=Count("attachments"))

    @admin.display(description=_("Attachments"), ordering="_attachments_count")
    def attachments_count(self, obj: Article) -> int:
        return getattr(obj, "_attachments_count", 0)


@admin.register(Category)
class CategoryAdmin(admin.ModelAdmin):
    list_display = ("title", "group", "order", "articles_count", "created", "updated")
    search_fields = ("title", "content")
    readonly_fields = ("created", "updated")
    ordering = ("-order", "title")
    date_hierarchy = "created"
    save_on_top = True
    list_per_page = 50
    list_filter = ("group",)
    autocomplete_fields = ("group",)
    list_select_related = ("group",)

    fieldsets = (
        (_("Appearance"), {"fields": ("image",)}),
        (_("Basics"), {"fields": ("title", "group", "order")}),
        (_("Description"), {"fields": ("content",)}),
        (_("Metadata"), {"fields": ("created", "updated"), "classes": ("collapse",)}),
    )

    def get_queryset(self, request):
        qs = super().get_queryset(request)
        return qs.annotate(_articles_count=Count("articles"))

    @admin.display(description="Liczba artykułów", ordering="_articles_count")
    def articles_count(self, obj: Category) -> int:
        return getattr(obj, "_articles_count", 0)


@admin.register(ArticleAttachment)
class ArticleAttachmentAdmin(admin.ModelAdmin):
    list_display = ("article", "type", "file", "created", "updated")
    list_filter = ("type",)
    search_fields = ("article__title",)
    readonly_fields = ("created", "updated")
    autocomplete_fields = ("article",)
    list_select_related = ("article",)
    ordering = ("-created",)
    date_hierarchy = "created"
    list_per_page = 50

    fieldsets = (
        (_("Relation"), {"fields": ("article",)}),
        (_("Attachment"), {"fields": ("file", "type")}),
        (_("Metadata"), {"fields": ("created", "updated"), "classes": ("collapse",)}),
    )


@admin.register(CategoryGroup)
class CategoryGroupAdmin(admin.ModelAdmin):
    list_display = ("title", "order", "categories_count", "created", "updated")
    search_fields = ("title",)
    readonly_fields = ("created", "updated")
    ordering = ("-order", "title")
    date_hierarchy = "created"
    save_on_top = True
    list_per_page = 50

    fieldsets = (
        (_("Basics"), {"fields": ("title", "order")}),
        (_("Extras"), {"fields": ("description", "icon")}),
        (_("Metadata"), {"fields": ("created", "updated"), "classes": ("collapse",)}),
    )

    def get_queryset(self, request):
        qs = super().get_queryset(request)
        return qs.annotate(_categories_count=Count("categories"))

    @admin.display(description=_("Categories"), ordering="_categories_count")
    def categories_count(self, obj: CategoryGroup) -> int:
        return getattr(obj, "_categories_count", 0)
