from django.db.models import ForeignKey
from django.db.models import CASCADE
from django.db.models import JSONField
from django.db.models import CharField
from django.db.models import TextChoices

from django.conf import settings

from table.models.base import BaseModel

class TableConfiguration(BaseModel):
    """
    Defines the TableConfig model for establishing relationships between users
    and tables in the system.

    Attributes:
        user (ManyToManyField<User>):
         Relationship between users and table configs.

        table (ManyToManyField<Table>):
         Relationship between tables and table configs.

        sort_field (CharField | null):
         Field to sort by. It's not possible to validate the value.

        sort_direction (TableConfig.SortDirection | null):
         Sort direction.

        columns_order (JSONField | null):
         List of columns ordered by the user.

        visible_columns (JSONField | null):
         List of visible columns.
         Columns should be sorted.
         If the value is null, frontend should use defaults.
    """
    class SortDirection(TextChoices):
        ASC = 'asc'
        DESC = 'desc'

    sort_field = CharField(max_length=128, null=True)
    sort_direction = CharField(max_length=128, choices=SortDirection.choices, null=True)
    visible_columns = JSONField(default=None, null=True)
    columns_order = JSONField(default=None, null=True)
    user = ForeignKey(to=settings.AUTH_USER_MODEL, related_name='table_configs', on_delete=CASCADE)
    table = ForeignKey(to='table.Table', related_name='table_configs', on_delete=CASCADE)

    def __str__(self) -> str:
        """
        Returns a string representation of the object.
        """
        return f"{repr(self.user)} - {self.table.key}"
