from django.conf import settings
from django.utils.translation import gettext_lazy as _
from django.db.models import QuerySet
from rest_framework.exceptions import ValidationError

from table.api.serializers import TableConfigurationSerializer
from table.api.serializers import UpdateTableConfigurationSerializer
from table.api.views.base import BaseViewSet
from table.models import Table
from table.models import TableConfiguration

from rest_framework.decorators import action
from rest_framework.response import Response
from rest_framework.exceptions import NotFound

# noinspection PyUnusedLocal
class TableConfigurationViewSet(BaseViewSet):
    """
    ViewSet for accessing TableConfig data.
    """
    serializer_class = TableConfigurationSerializer
    update_serializer_class = UpdateTableConfigurationSerializer

    def get_queryset(self) -> QuerySet[TableConfiguration]:
        """
        Retrieves and returns the queryset for the TableConfig model.
        """
        queryset = TableConfiguration.objects.select_related('user')

        return queryset.filter(
            user=self.request.user,
        )

    def get_serializer_class(self):
        """
        Determines and returns the appropriate serializer class based on the current action.

        If the current action matches the name of the `update_configuration_view` method, it
        returns the `update_serializer_class`. Otherwise, it falls back to the default behavior
        of obtaining the serializer class from the superclass.
        """
        if self.action == self.update_configuration_view.__name__:
            return self.update_serializer_class

        if self.action == self.get_configuration_view.__name__:
            return self.serializer_class

        raise ValidationError(
            _("No serializer class found for the action. Please use only allowed actions on this view set.")
        )

    def _get_key(self) -> str:
        """
        Retrieves the key from the URL kwargs.
        """
        return self.kwargs[self.lookup_url_kwarg]

    def _get_table(self) -> Table:
        """
        Fetches or creates a ``Table`` object based on the provided key.

        :raises IntegrityError: If the table key is not unique.
        """
        key = self._get_key()
        table, _ = Table.objects.get_or_create(key=key)
        return table

    def get_object(self) -> TableConfiguration:
        """
        Retrieves or creates an object from the queryset filtered by the specified table key.

        The function first filters the queryset obtained from `get_queryset` using the method
        `filter_queryset`. It then attempts to retrieve or create an object using the provided
        key in the URL and finally checks the permissions for the object before returning it.

        Returns:
            obj: The retrieved or newly created object.

        Raises:
            DoesNotExist: If the object cannot be found in the database, and creation fails.
            PermissionDenied: If the request lacks proper permissions for the object.
        """
        assert self.request.user is not None
        queryset = self.filter_queryset(self.get_queryset())
        obj, _ = queryset.get_or_create(table=self._get_table(), user=self.request.user)
        self.check_object_permissions(self.request, obj)
        return obj

    def get_object_or_404(self, key, queryset: QuerySet[TableConfiguration] = None, user: settings.AUTH_USER_MODEL = None):
        """
        Retrieve an object from the database using the provided key, queryset, and user. If the object is
        not found, an exception is raised. Additionally, it verifies the permissions for the retrieved object.
        """
        if queryset is None:
            queryset = self.get_queryset()

        if user is None:
            user = self.request.user

        try:
            obj = queryset.get(table__key=key, user=user)
            self.check_object_permissions(self.request, obj)
            return obj

        except TableConfiguration.DoesNotExist:
            raise NotFound()

    # noinspection PyUnusedLocal
    @action(detail=True, methods=["GET"], url_path="config")
    def get_configuration_view(self, request, *args, **kwargs) -> Response:
        """
        Handles HTTP GET requests to retrieve the configuration details for a specific
        object using the detail view.
        """
        serializer = self.get_serializer(instance=self.get_object())
        return Response(data=serializer.data)

    @action(detail=True, methods=["PUT"], url_path="update")
    def update_configuration_view(self, request, *args, **kwargs) -> Response:
        """
        Handles the update of a specific configuration via a PUT request. Retrieves the
        configuration object, validates the provided data through the serializer, and
        then updates the object in the system.
        """
        key = self.kwargs[self.lookup_url_kwarg]
        queryset = self.get_queryset()
        obj = self.get_object_or_404(key=key)
        serializer = self.get_serializer(instance=obj, data=request.data)
        serializer.is_valid(raise_exception=True)
        serializer.update(instance=obj, validated_data=serializer.validated_data)
        return Response(data=serializer.data)
