from django.utils.translation import gettext_lazy as _

from rest_framework.serializers import CharField
from rest_framework.serializers import ChoiceField
from rest_framework.serializers import JSONField
from rest_framework.exceptions import ValidationError

from table.api.serializers.base import BaseReadSerializer
from table.api.serializers.base import BaseUpdateSerializer

from table.models import TableConfiguration


class TableConfigurationSerializer(
    BaseReadSerializer,
):
    table_key = CharField(source="table.key", read_only=True)

    class Meta:
        model = TableConfiguration
        fields = [
            "sort_field",
            "sort_direction",
            "visible_columns",
            "table_key",
            "columns_order",
        ]

class UpdateTableConfigurationSerializer(BaseUpdateSerializer):
    representation = TableConfigurationSerializer

    sort_field = CharField(required=False, allow_null=True)
    sort_direction = ChoiceField(required=False, choices=TableConfiguration.SortDirection.choices, allow_null=True)
    visible_columns = JSONField(required=False, allow_null=False)
    columns_order = JSONField(required=False, allow_null=False)

    @staticmethod
    def _check_string_array(value):
        """
        Checks if the value is a list of strings.
        Raises ValidationError if not.
        """
        if not isinstance(value, list):
            raise ValidationError(
                _("Array must be a list.")
            )

        for item in value:
            if not isinstance(item, str):
                raise ValidationError(
                    _(f"Array must be a list of strings. "
                      f"But item `{item}` at {value.index(item)} position is not a string.")
                )

    @staticmethod
    def _is_array_items_unique(value):
        """
        Checks if the items in the array are unique.
        Returns true or false.
        """
        return len(value) == len(set(value))

    def validate_visible_columns(self, value):
        """
        Validate `visible_columns` field.
        """
        self._check_string_array(value)

        if not self._is_array_items_unique(value):
            raise ValidationError(
                _("`visible_columns` must contain unique items.")
            )

        return value

    def validate_columns_order(self, value):
        """
        Validate `columns_order` field.
        """
        self._check_string_array(value)

        if not self._is_array_items_unique(value):
            raise ValidationError(
                _("`columns_order` must contain unique items.")
            )

        return value

    class Meta:
        model = TableConfiguration
        fields = [
            "sort_field",
            "sort_direction",
            "visible_columns",
            "columns_order",
        ]
