from rest_framework.serializers import DateTimeField
from rest_framework.serializers import ModelSerializer
from rest_framework.serializers import IntegerField

class BaseReadSerializer(ModelSerializer):
    """
    Basic "Read" serializer.
    Adds id, created, and updated fields.
    """

    def get_fields(self):
        fields = super().get_fields()
        fields['id'] = IntegerField(read_only=True)
        fields['created'] = DateTimeField(read_only=True)
        fields['updated'] = DateTimeField(read_only=True)
        return fields

class BaseCreateSerializer(ModelSerializer):
    """
    Basic "Create" serializer.

    Args:
        representation (class):
            Serializer class for representation.
            Must be set, otherwise to_representation() will raise an AssertionError.
    """
    representation = None

    def to_representation(self, instance):
        assert self.representation is not None, "representation class must be set."
        return self.representation(instance).data

    def update(self, instance, validated_data):
        raise PermissionError(
            "You cannot update using this serializer."
            " Please use serializer for updating."
        )

class BaseUpdateSerializer(ModelSerializer):
    """
    Basic "Update" serializer.

    Args:
        representation (class):
            Serializer class for representation.
            Must be set, otherwise to_representation() will raise an AssertionError.
    """
    representation = None

    def to_representation(self, instance):
        assert self.representation is not None, "representation class must be set."
        return self.representation(instance).data

    def create(self, validated_data):
        raise PermissionError(
            "You cannot create using this serializer."
            " Please use serializer for creating."
        )