from django.contrib import admin
from django.db.models import Count
from django.utils.translation import gettext_lazy as _

from table.models import Table
from table.models import TableConfiguration


class TableConfigurationInline(admin.TabularInline):
    model = TableConfiguration
    extra = 0
    show_change_link = True
    raw_id_fields = ('user',)
    fields = ('user', 'sort_field', 'sort_direction', 'visible_columns')
    verbose_name = _('Configuration')
    verbose_name_plural = _('Configurations')


@admin.register(Table)
class TableAdmin(admin.ModelAdmin):
    list_display = ('id', 'title', 'key', 'configs_count')
    search_fields = ('title', 'key')
    ordering = ('key',)
    list_per_page = 50
    inlines = [TableConfigurationInline]

    def get_queryset(self, request):
        qs = super().get_queryset(request)
        return qs.annotate(_configs_count=Count('table_configs'))

    @admin.display(description=_('Configs'))
    def configs_count(self, obj):
        return getattr(obj, '_configs_count', obj.table_configs.count())


@admin.register(TableConfiguration)
class TableConfigurationAdmin(admin.ModelAdmin):
    list_display = (
        'id',
        'table',
        'user',
        'sort_field',
        'sort_direction',
        'visible_columns_display',
    )
    list_filter = ('sort_direction', 'table')
    search_fields = ('table__title', 'table__key', 'user__id')
    ordering = ('table', 'user')
    list_select_related = ('table', 'user')
    list_per_page = 50
    raw_id_fields = ('user', 'table')

    def visible_columns_display(self, obj):
        data = obj.visible_columns or []
        if isinstance(data, (list, tuple)):
            preview = ', '.join(map(str, data[:5]))
            if len(data) > 5:
                preview += f' … (+{len(data) - 5})'
            return preview or '-'
        return str(data) if data is not None else '-'

    visible_columns_display.short_description = _('Visible columns')
