"""
Tests for the TextProcessor class.
"""

import pytest
from example_package import TextProcessor


class TestTextProcessor:
    """Test cases for TextProcessor class."""
    
    def setup_method(self):
        """Set up test fixtures."""
        self.processor = TextProcessor()
    
    def test_clean_text(self):
        """Test text cleaning functionality."""
        dirty_text = "  Hello   World  \n\t  "
        cleaned = self.processor.clean_text(dirty_text)
        assert cleaned == "Hello World"
        
        # Test with already clean text
        clean_text = "Hello World"
        result = self.processor.clean_text(clean_text)
        assert result == "Hello World"
    
    def test_count_words(self):
        """Test word counting functionality."""
        assert self.processor.count_words("Hello World") == 2
        assert self.processor.count_words("One") == 1
        assert self.processor.count_words("") == 0
        assert self.processor.count_words("   ") == 0
        assert self.processor.count_words("Hello World Test") == 3
    
    def test_reverse_text(self):
        """Test text reversal functionality."""
        assert self.processor.reverse_text("Hello") == "olleH"
        assert self.processor.reverse_text("123") == "321"
        assert self.processor.reverse_text("") == ""
        assert self.processor.reverse_text("a") == "a"
    
    def test_extract_numbers(self):
        """Test number extraction functionality."""
        text = "I have 5 apples and 10 oranges"
        numbers = self.processor.extract_numbers(text)
        assert numbers == ["5", "10"]
        
        # Test with no numbers
        text_no_numbers = "No numbers here"
        numbers = self.processor.extract_numbers(text_no_numbers)
        assert numbers == []
        
        # Test with multiple numbers
        text_multiple = "1 2 3 4 5"
        numbers = self.processor.extract_numbers(text_multiple)
        assert numbers == ["1", "2", "3", "4", "5"]
    
    def test_to_title_case(self):
        """Test title case conversion."""
        assert self.processor.to_title_case("hello world") == "Hello World"
        assert self.processor.to_title_case("HELLO WORLD") == "Hello World"
        assert self.processor.to_title_case("hELLo WoRLd") == "Hello World"
        assert self.processor.to_title_case("") == ""
    
    def test_processing_history(self):
        """Test processing history functionality."""
        self.processor.clean_text("test")
        self.processor.count_words("hello world")
        
        history = self.processor.get_processing_history()
        assert len(history) == 2
        assert "Cleaned: test" in history
        assert "Word count: 2" in history
    
    def test_clear_history(self):
        """Test clearing processing history."""
        self.processor.clean_text("test")
        self.processor.clear_history()
        
        history = self.processor.get_processing_history()
        assert len(history) == 0
    
    def test_complex_text_processing(self):
        """Test complex text processing scenarios."""
        text = "  Hello   World 123  "
        
        # Clean the text
        cleaned = self.processor.clean_text(text)
        assert cleaned == "Hello World 123"
        
        # Count words
        word_count = self.processor.count_words(cleaned)
        assert word_count == 3
        
        # Extract numbers
        numbers = self.processor.extract_numbers(text)
        assert numbers == ["123"]
        
        # Convert to title case
        title_case = self.processor.to_title_case(cleaned)
        assert title_case == "Hello World 123"
        
        # Reverse text
        reversed_text = self.processor.reverse_text(cleaned)
        assert reversed_text == " 321 dlroW olleH"
