"""
Text utilities module providing text processing functions.
"""

import re
from typing import List


class TextProcessor:
    """
    A class for processing text with various utility functions.
    """
    
    def __init__(self):
        """Initialize the text processor."""
        self.processed_texts = []
    
    def clean_text(self, text: str) -> str:
        """
        Clean text by removing extra whitespace and normalizing.
        
        Args:
            text: Input text to clean
            
        Returns:
            Cleaned text
        """
        # Remove extra whitespace and normalize
        cleaned = re.sub(r'\s+', ' ', text.strip())
        self.processed_texts.append(f"Cleaned: {cleaned}")
        return cleaned
    
    def count_words(self, text: str) -> int:
        """
        Count words in text.
        
        Args:
            text: Input text
            
        Returns:
            Number of words
        """
        words = len(text.split())
        self.processed_texts.append(f"Word count: {words}")
        return words
    
    def reverse_text(self, text: str) -> str:
        """
        Reverse the text.
        
        Args:
            text: Input text
            
        Returns:
            Reversed text
        """
        reversed_text = text[::-1]
        self.processed_texts.append(f"Reversed: {reversed_text}")
        return reversed_text
    
    def extract_numbers(self, text: str) -> List[str]:
        """
        Extract all numbers from text.
        
        Args:
            text: Input text
            
        Returns:
            List of found numbers as strings
        """
        numbers = re.findall(r'\d+', text)
        self.processed_texts.append(f"Found numbers: {numbers}")
        return numbers
    
    def to_title_case(self, text: str) -> str:
        """
        Convert text to title case.
        
        Args:
            text: Input text
            
        Returns:
            Text in title case
        """
        title_text = text.title()
        self.processed_texts.append(f"Title case: {title_text}")
        return title_text
    
    def get_processing_history(self) -> List[str]:
        """
        Get processing history.
        
        Returns:
            List of processing operations
        """
        return self.processed_texts.copy()
    
    def clear_history(self) -> None:
        """Clear processing history."""
        self.processed_texts.clear()
