import os

from enum import Enum


class File(Enum):
    FILE_0 = "0.epp"
    FILE_1 = "1.epp"
    FILE_2 = "2.epp"
    MANY = "many.epp"


class Fixtures:
    def __init__(self):
        self.base_path = os.path.join(os.path.dirname(__file__))

    def get_path(self, file: File) -> str:
        """
        Constructs a full file path by joining the base path with the given
        filename. If the constructed path does not exist, it raises a
        FileNotFoundError.

        :param file: Name of the file to construct the path for.
        :type file: Str is
        :return: The full path to the specified file if it exists.
        :rtype: Str
        :raises FileNotFoundError: If the constructed path does not exist.
        """
        filename = file.value
        path = os.path.join(self.base_path, filename)
        if not os.path.exists(path):
            raise FileNotFoundError(
                f"File '{filename}' does not exist in directory {self.base_path}"
            )
        return path

    def read_text(self, filename: File, encoding="utf-8") -> str:
        """
        Reads the content of a text file located at the specified path, using the
        provided encoding. Returns the file's content as a string.

        :param filename: The file to be read, represented as a TestFile object.
        :param encoding: The text encoding to be used for reading the file. Defaults to "utf-8".
        :type encoding: Str, optional
        :return: The content of the file as a string.
        :rtype: Str
        """
        with open(self.get_path(filename), "r", encoding=encoding) as f:
            return f.read()

    def read_bytes(self, filename: File) -> bytes:
        with open(self.get_path(filename), "rb") as f:
            return f.read()


def read_test_file(filename: File, **kwargs) -> str:
    """Read a test file from the fixtures' directory."""
    return Fixtures().read_text(filename, **kwargs)
