import pytest
from xml.etree.ElementTree import tostring
from app.domain.edi.models import Seller


@pytest.fixture
def sample_seller():
    return Seller(
        iln="5903293144008",
        account_number="84 1160 2202 0000 0003 4391 9000",
        tax_id="5223112460",
        name="Big Nature Sp. z o.o.",
        street_and_number="Warszawska 12A",
        city_name="Palmiry",
        postal_code="05-152",
        country="PL",
        code_by_buyer=None,
        court_and_capital_information=None,
    )


def test_to_xml_returns_element(sample_seller):
    xml_el = sample_seller.to_xml()
    assert xml_el.tag == "Seller", "Root element should be <Seller>"
    expected_tags = {
        "ILN",
        "AccountNumber",
        "TaxID",
        "CodeByBuyer",
        "Name",
        "StreetAndNumber",
        "CityName",
        "PostalCode",
        "Country",
        "CourtAndCapitalInformation",
        "UtilizationRegisterNumber",
    }
    actual_tags = {child.tag for child in xml_el}
    assert expected_tags == actual_tags, "Missing or extra tags in Seller XML"


def test_to_xml_contains_expected_values(sample_seller):
    xml_str = tostring(sample_seller.to_xml(), encoding="unicode")
    assert "<ILN>5903293144008</ILN>" in xml_str
    assert "<AccountNumber>84 1160 2202 0000 0003 4391 9000</AccountNumber>" in xml_str
    assert "<TaxID>5223112460</TaxID>" in xml_str
    assert "<Name>Big Nature Sp. z o.o.</Name>" in xml_str
    assert "<StreetAndNumber>Warszawska 12A</StreetAndNumber>" in xml_str
    assert "<CityName>Palmiry</CityName>" in xml_str
    assert "<PostalCode>05-152</PostalCode>" in xml_str
    assert "<Country>PL</Country>" in xml_str


def test_to_xml_handles_none_values():
    seller = Seller(
        iln="123",
        account_number="456",
        tax_id="789",
        name="Test",
        street_and_number="Street 1",
        city_name="City",
        postal_code="00-000",
        country="PL",
        code_by_buyer=None,
        court_and_capital_information=None,
    )
    xml_str = tostring(seller.to_xml(), encoding="unicode")
    # pola opcjonalne mają być puste, ale istnieć
    assert "<CodeByBuyer />" in xml_str or "<CodeByBuyer></CodeByBuyer>" in xml_str
    assert (
        "<CourtAndCapitalInformation />" in xml_str
        or "<CourtAndCapitalInformation></CourtAndCapitalInformation>" in xml_str
    )
