from dataclasses import dataclass
from argparse import ArgumentParser

from .constants import DEFAULT_KEEP_FILES
from .constants import DEFAULT_LOG_LEVEL
from .constants import DEFAULT_LOG_FILE

parser = ArgumentParser(
    description="EPP → XML converter for Nexo-Pro integration",
)

subparsers = parser.add_subparsers(dest="command", required=True, description="Komenda do wykonania.")

# Subcommands
# --------------------------------
sub_run = subparsers.add_parser("run", help="Uruchom program.")
sub_test = subparsers.add_parser("test", help="Test.")
sub_watch = subparsers.add_parser("watch", help="Monitoruj zmiany w folderze.")
sub_work = subparsers.add_parser("work", help="Przetwarzaj zmiany w folderze w określonych interwałach.")
sub_work.add_argument(
    "-i",
    "--interval",
    type=int,
    help="Określ interwał czasowy, w sekundach, w którym ma się wykonać przetwarzanie zmiany w folderze.",
)
sub_init = subparsers.add_parser("init", help="Zainicjuj folder z danymi i konfigurację.")
sub_add = subparsers.add_parser("add-nip", help="Dodaj NIP firmy do konfiguracji.")

# Configuration
# --------------------------------
parser.add_argument(
    "-c",
    "--config",
    type=str,
    help="Ścieżka do pliku konfiguracyjnego (domyślnie: ./config.json)",
)
parser.add_argument(
    "--keep-files",
    default=DEFAULT_KEEP_FILES,
    action="store_true",
    help="Do not move input files after processing.",
)

# Logging
# --------------------------------
parser.add_argument(
    "-l",
    "--log-level",
    default=DEFAULT_LOG_LEVEL,
    choices=["TRACE", "DEBUG", "INFO", "SUCCESS", "WARNING", "ERROR", "CRITICAL"],
    help="Stopień logowania (domyślnie: INFO lub wartość z LOG_LEVEL)",
)
parser.add_argument(
    "-lf",
    "--log-file",
    type=str,
    help=f"Ścieżka do pliku logowania (domyślnie: {DEFAULT_LOG_FILE})",
    required=False,
    default=DEFAULT_LOG_FILE,
)

@dataclass
class Args:
    command: str
    config: str
    log_level: str
    log_file: str
    keep_files: bool
    work_interval_seconds: int | None

def parse_args() -> Args:
    args = parser.parse_args()
    return Args(
        command=args.command,
        config=args.config,
        log_level=args.log_level,
        log_file=args.log_file,
        keep_files=args.keep_files,
        work_interval_seconds=getattr(args, 'interval', None),
    )