import shutil
from pathlib import Path

from loguru import logger
from slugify import slugify

from app.domain.config import Config
from app.domain.converters import ConvertError
from app.domain.converters import Converter
from app.domain.converters import ILNError
from app.domain.edi.models import Invoice
from app.domain.edi.models import InvoiceHeader
from app.domain.epp.models import Document
from app.domain.epp.parser import Parser
from app.infra.exceptions import QuietException
from app.infra.parser import Args


class FileProcessorError(QuietException): ...

class FileProcessor:
    """ Utility to process a single file. """
    def __init__(self, config: Config, args: Args):
        self.config = config
        self.args = args
        self.parser = Parser()
        self.converter = Converter(config)

    def _get_target_xml_path(self, inv: InvoiceHeader) -> Path:
        """ Get target XML for an invoice. """
        slug = slugify(inv.invoice_number, lowercase=False, separator="_")
        file_name = f"{slug}.xml"
        return Path(self.config.output_dir) / file_name

    def _save_invoice_xml(self, invoice: Invoice):
        """ Save invoice XML into the appropriate file. """
        path = self._get_target_xml_path(invoice.header)
        with open(path, mode="w") as f:
            f.write(invoice.to_xml_string())

    def _move_epp_file(self, file: Path):
        """ Move EPP file to processed directory. """
        if self.args.keep_files:
            logger.debug(f"Keeping file {file.name} in input directory.")
        new_path = Path(self.config.processed_dir) / file.name
        shutil.move(file, new_path)

    def _process_invoice(self, invoice: Invoice):
        """ Process a single invoice. """
        self._save_invoice_xml(invoice)

    def _process_all_invoices(self, file: Path, invoices: list[Invoice]):
        """ Process all invoices in a single file. """
        if len(invoices) == 1:  # What is the idea behind?
            invoice = invoices[0]
            self._process_invoice(invoice)
            return

        for invoice in invoices:
            self._process_invoice(invoice)

        self._move_epp_file(file)

    def process_file(self, file: Path) -> list[Invoice]:
        """ Process a single file. """
        try:
            with open(file, mode="r", encoding="cp1250") as f:
                epp_file_content = f.read()

            document: Document = self.parser.parse(epp_file_content)
            invoices = self.converter.convert_all(document)
            self._process_all_invoices(file, invoices)
            return invoices
        except ILNError as e:
            logger.error(f"Missing ILN in configuration. Please add NIP: {e.nip}. You can use `epp add nip {e.nip}` to add it.")
            raise SystemExit(1)
        except ConvertError as e:
            logger.error(f"Could not process file {file.name}: {e}")
            raise FileProcessorError() from e