from app.domain.epp.models import (
    Document,
    DocumentHeader,
    Header,
    Item,
    PriceListItem,
    LineItem,
    Contractor,
    DeliveryCompletionDate,
)
from .epp_parser import parse_epp, EPP


class ParserException(Exception):
    pass


class Parser:
    """Parser class to parse EPP files content into models."""

    def __init__(self):
        pass

    @staticmethod
    def get_document_headers(epp: EPP):
        headers = epp.get_headers()
        for header in headers:
            yield DocumentHeader.from_csv(header)

    def parse(self, file_content: str) -> Document:
        """Parse the EPP file content and return an object."""
        epp: EPP = parse_epp(file_content)
        price_list_items = [
            PriceListItem.from_csv(row) for row in epp.headers.get("CENNIK", [])
        ]
        items = [Item.from_csv(row) for row in epp.headers.get("TOWARY", [])]
        line_items = [
            LineItem.from_csv(row) for row in epp.headers.get(epp.get_header(), [])
        ]
        contractors = [
            Contractor.from_csv(row) for row in epp.headers.get("KONTRAHENCI", [])
        ]
        delivery_completion_date = [
            DeliveryCompletionDate.from_csv(row)
            for row in epp.headers.get("DATYZAKONCZENIA", [])
        ]
        return Document(
            file_header=Header.from_csv(epp.info),
            document_headers=list(self.get_document_headers(epp)),
            items=items,
            price_list_items=price_list_items,
            line_items=line_items,
            contractors=contractors,
            delivery_completion_date=delivery_completion_date,
        )
