from datetime import date

from app.domain.epp import parse_date_or_null


class DocumentHeader:
    """
    Represents the header of a turnover or warehouse document.

    Attributes
    ----------
    doc_type : str
        (1) Typ dokumentu – type of document (e.g. "FZ", "FS", "WZ", etc.).
    status : int
        (2) Status dokumentu – status of document (0 = deferred, 1 = executed, 2 = cancelled, 3 = booked).
    fiscal_registration_status : int
        (3) Status rejestracji fiskalnej dokumentu (flaga) – fiscal registration flag/status.
    document_number : int
        (4) Numer dokumentu – document number (long integer).
    supplier_document_number : str
        (5) Numer dokumentu dostawcy – supplier’s document number (text up to 20).
    user_number_extension : str
        (6) Rozszerzenie numeru wpisane przez użytkownika – extension of number entered by user (text up to 10).
    full_document_number : str
        (7) Pełny numer dokumentu – full document number (text up to 30).
    corrected_document_number : str
        (8) Numer dokumentu korygowanego – number of corrected documents (text up to 30).
    corrected_document_date : str
        (9) Data wystawienia dokumentu korygowanego – date of issuance of a corrected document.
    order_number : str
        (10) Numer zamówienia – order number (text up to 30).
    destination_warehouse : str
        (11) Magazyn docelowy dla MM (symbol) – destination warehouse for inter-warehouse movement (text up to 3).
    customer_id : str
        (12) Kod identyfikacyjny kontrahenta – customer/contractor identification code (text up to 20).
    customer_short_name : str
        (13) Nazwa skrócona kontrahenta – customer short name (text up to 40).
    customer_full_name : str
        (14) Nazwa pełna kontrahenta – customer full name (text up to 255).
    customer_city : str
        (15) Miasto kontrahenta – customer city (text up to 30).
    customer_postal_code : str
        (16) Kod pocztowy kontrahenta – customer postal code (text up to 6).
    customer_address : str
        (17) Ulica i numer kontrahenta (adres) – customer street and number (text up to 50).
    customer_nip : str
        (18) NIP kontrahenta (unijny lub krajowy) – customer tax identification number (text up to 20).
    category_name : str
        (19) Kategoria (nazwa) – category name (text up to 30).
    category_subtitle : str
        (20) Podtytuł kategorii – category subtitle (text up to 50).
    place_of_issue : str
        (21) Miejsce wystawienia – place of issue (text up to 30).
    issue_date : date
        (22) Data wystawienia – issue date.
    sale_date : str
        (23) Data sprzedaży – sale date.
    receipt_date : str
        (24) Data otrzymania – date of receipt.
    number_of_items : int
        (25) Liczba pozycji – number of items (long integer).
    net_price_document : bool
        (26) Czy dokument wystawiany wg cen netto – whether document issued using net prices (logical).
    active_price_name : str
        (27) Aktywna cena (nazwa) – active price name (text up to 20).
    net_value : float
        (28) Wartość netto – net value (amount).
    vat_value : float
        (29) Wartość VAT – VAT value (amount).
    gross_value : float
        (30) Wartość brutto – gross value (amount).
    cost_value : float
        (31) Koszt – cost (amount).
    discount_name : str
        (32) Rabat (nazwa) – discount name (text up to 30).
    discount_percent : float
        (33) Rabat (procent) – discount percentage (amount).
    payment_method_name : str
        (34) Forma płatności (nazwa) – payment method name (text up to 30).
    payment_due_date : date
        (35) Termin płatności – payment due date (date).
    amount_paid_on_delivery : float
        (36) Kwota zapłacona przy odbiorze dokumentu – amount paid on delivery (amount).
    amount_due : float
        (37) Wartość do zapłaty – amount due (amount).
    rounding_unit : int
        (38) Zaokrąglenie wartości do zapłaty – rounding unit (0=1gr,1=10gr,2=1zł).
    rounding_vat : int
        (39) Zaokrąglenie wartości VAT – rounding VAT (0=1gr,1=10gr,2=1zł).
    auto_vat_table_calc : bool
        (40) Automatycznie przeliczana tabela VAT i wartość dokumentu – auto calc VAT table and doc value (logical).
    extended_statuses : int
        (41) Statusy rozszerzone i specjalne dokumentów – extended and special statuses (byte).
    currency : str
        (47) Waluta (symbol) - currency symbol.
    remarks : str
        (49): Uwagi - remarks
    country_symbol : str
        (61): Prefiks państwa UE kontrahenta – country symbol.
    """

    def __init__(
        self,
        doc_type: str,
        status: int,
        fiscal_registration_status: int,
        document_number: int,
        supplier_document_number: str,
        user_number_extension: str,
        full_document_number: str,
        corrected_document_number: str,
        corrected_document_date: str,
        order_number: str,
        destination_warehouse: str,
        customer_id: str,
        customer_short_name: str,
        customer_full_name: str,
        customer_city: str,
        customer_postal_code: str,
        customer_address: str,
        customer_nip: str,
        category_name: str,
        category_subtitle: str,
        place_of_issue: str,
        issue_date: date,
        sale_date: date,
        receipt_date: date,
        number_of_items: int,
        net_price_document: bool,
        active_price_name: str,
        net_value: float,
        vat_value: float,
        gross_value: float,
        cost_value: float,
        discount_name: str,
        discount_percent: float,
        payment_method_name: str,
        payment_due_date: date,
        amount_paid_on_delivery: float,
        amount_due: float,
        rounding_unit: int,
        rounding_vat: int,
        auto_vat_table_calc: bool,
        extended_statuses: int,
        currency: str,
        remarks: str | None = None,
        country_symbol: str | None = None,
    ):
        self.doc_type = doc_type
        self.status = status
        self.fiscal_registration_status = fiscal_registration_status
        self.document_number = document_number
        self.supplier_document_number = supplier_document_number
        self.user_number_extension = user_number_extension
        self.full_document_number = full_document_number
        self.corrected_document_number = corrected_document_number
        self.corrected_document_date = corrected_document_date
        self.order_number = order_number
        self.destination_warehouse = destination_warehouse
        self.customer_id = customer_id
        self.customer_short_name = customer_short_name
        self.customer_full_name = customer_full_name
        self.customer_city = customer_city
        self.customer_postal_code = customer_postal_code
        self.customer_address = customer_address
        self.customer_nip = customer_nip
        self.category_name = category_name
        self.category_subtitle = category_subtitle
        self.place_of_issue = place_of_issue
        self.issue_date = issue_date
        self.sale_date = sale_date
        self.receipt_date = receipt_date
        self.number_of_items = number_of_items
        self.net_price_document = net_price_document
        self.active_price_name = active_price_name
        self.net_value = net_value
        self.vat_value = vat_value
        self.gross_value = gross_value
        self.cost_value = cost_value
        self.discount_name = discount_name
        self.discount_percent = discount_percent
        self.payment_method_name = payment_method_name
        self.payment_due_date = payment_due_date
        self.amount_paid_on_delivery = amount_paid_on_delivery
        self.amount_due = amount_due
        self.rounding_unit = rounding_unit
        self.rounding_vat = rounding_vat
        self.auto_vat_table_calc = auto_vat_table_calc
        self.extended_statuses = extended_statuses
        self.currency = currency
        self.remarks = remarks
        self.country_symbol = country_symbol

    def get_full_number(self) -> str:
        """
        Return the full document number.
        :return: str
        """
        return f"{self.doc_type} {self.full_document_number}"

    @staticmethod
    def from_csv(data: list) -> "DocumentHeader":
        if len(data) < 41:
            raise ValueError(
                f"Expected at least 41 values to build DocumentHeader, got {len(data)}"
            )

        def to_str(v):
            return None if v is None else str(v).strip()

        def to_int(v):
            s = str(v).strip()
            if s == "":
                return 0
            return int(float(s)) if any(ch in s for ch in (".", "e", "E")) else int(s)

        def to_float(v):
            s = str(v).strip()
            if s == "":
                return 0.0
            return float(s)

        def to_bool(v):
            if isinstance(v, bool):
                return v
            s = str(v).strip().lower()
            if s in {"1", "t", "true", "yes", "y"}:
                return True
            if s in {"0", "f", "false", "no", "n", ""}:
                return False
            try:
                return bool(int(s))
            except Exception as exc:
                raise ValueError(f"Cannot convert value to bool: {v!r}") from exc

        return DocumentHeader(
            to_str(data[0]),
            to_int(data[1]),
            to_int(data[2]),
            to_int(data[3]),
            to_str(data[4]),
            to_str(data[5]),
            to_str(data[6]),
            to_str(data[7]),
            to_str(data[8]),
            to_str(data[9]),
            to_str(data[10]),
            to_str(data[11]),
            to_str(data[12]),
            to_str(data[13]),
            to_str(data[14]),
            to_str(data[15]),
            to_str(data[16]),
            to_str(data[17]),
            to_str(data[18]),
            to_str(data[19]),
            to_str(data[20]),
            parse_date_or_null(data[21]),
            parse_date_or_null(data[22]),
            parse_date_or_null(data[23]),
            to_int(data[24]),
            to_bool(data[25]),
            to_str(data[26]),
            to_float(data[27]),
            to_float(data[28]),
            to_float(data[29]),
            to_float(data[30]),
            to_str(data[31]),
            to_float(data[32]),
            to_str(data[33]),
            parse_date_or_null(data[34]),
            to_float(data[35]),
            to_float(data[36]),
            to_int(data[37]),
            to_int(data[38]),
            to_bool(data[39]),
            to_int(data[40]),
            currency=to_str(data[46]),
            remarks=to_str(data[48]),
            country_symbol=to_str(data[60]) if len(data) >= 61 else None,
        )
