from .epp_others import DeliveryCompletionDate
from .epp_contractor import Contractor
from .epp_line_item import LineItem
from .epp_price_list_item import PriceListItem
from .epp_document_header_model import DocumentHeader
from .epp_header import Header
from .epp_item import Item


class Document:
    """
    Represents a document in the EPP communication file.

    Attributes:
        file_header: Header instance from the [INFO] section.
    """

    def __init__(
        self,
        file_header: Header,
        document_headers: list[DocumentHeader],
        items: list[Item],
        price_list_items: list[PriceListItem],
        line_items: list[LineItem],
        contractors: list[Contractor],
        delivery_completion_date: list[DeliveryCompletionDate],
    ):
        self.file_header = file_header
        self.document_headers = document_headers
        self.items = items
        self.price_list_items = price_list_items
        self.line_items = line_items
        self.contractors = contractors
        self.delivery_completion_date = delivery_completion_date

    def get_price(self, item_id: str) -> PriceListItem | None:
        """Returns the price for the given item."""
        for item in self.price_list_items:
            if item.item_id == item_id:
                return item
        return None

    def get_barcode(self, item_id: str) -> str | None:
        """Returns the barcode for the given item."""
        for item in self.items:
            if item.item_id == item_id:
                return item.barcode
        return None

    def get_item(self, item_id: str) -> Item | None:
        """
        Returns the item for the given item_id.
        """
        for item in self.items:
            if item.item_id == item_id:
                return item
        return None

    def get_contractor(self, contractor_id: str) -> Contractor | None:
        for contractor in self.contractors:
            if contractor_id == contractor.identifier_code:
                return contractor
        return None

    # noinspection PyMethodMayBeStatic
    def get_full_number(self, document_header: DocumentHeader) -> str:
        """
        Get the full number of the document in the following format: '{doc_type} {full_number}'
        Returns:
            FS 2369/MAG/10/2025
        """
        return document_header.get_full_number()

    def get_delivery_completion_date(
        self, document_header: DocumentHeader
    ) -> DeliveryCompletionDate | None:
        full_number = self.get_full_number(document_header)
        for delivery_completion_date in self.delivery_completion_date:
            if delivery_completion_date.document_full_id == full_number:
                return delivery_completion_date
        return None
