from xml.etree.ElementTree import Element, SubElement

from .edi_tax_summary_line import TaxSummaryLine


class InvoiceSummary:
    def __init__(
        self,
        total_lines: int,
        total_net_amount: float,
        total_taxable_basis: int,
        total_tax_amount: float,
        total_gross_amount: float,
        tax_summary_lines: list[TaxSummaryLine],
        _validate: bool = True,
    ):
        if _validate:
            if not isinstance(total_lines, int) or total_lines < 0:
                raise ValueError("total_lines must be a non-negative integer")
            if not isinstance(total_net_amount, (int, float)) or total_net_amount < 0:
                raise ValueError("total_net_amount must be a non-negative number")
            if (
                not isinstance(total_taxable_basis, (int, float))
                or total_taxable_basis < 0
            ):
                raise ValueError("total_taxable_basic must be a non-negative number")
            if not isinstance(total_tax_amount, (int, float)) or total_tax_amount < 0:
                raise ValueError("total_tax_amount must be a non-negative number")
            if (
                not isinstance(total_gross_amount, (int, float))
                or total_gross_amount < 0
            ):
                raise ValueError("total_gross_amount must be a non-negative number")
            if not isinstance(tax_summary_lines, list) or not all(
                isinstance(x, TaxSummaryLine) for x in tax_summary_lines
            ):
                raise ValueError(
                    "tax_summary_lines must be a list of TaxSummaryLine objects"
                )

        self.total_lines = total_lines
        self.total_net_amount = total_net_amount
        self.total_tax_amount = total_tax_amount
        self.total_taxable_basis = total_taxable_basis
        self.total_gross_amount = total_gross_amount
        self.tax_summary = tax_summary_lines

    def to_xml(self) -> Element:
        """Return XML Element <Summary> including nested <TaxSummary>."""
        root = Element("Invoice-Summary")

        def _add(tag, value):
            el = SubElement(root, tag)
            el.text = str(value).strip() if value is not None else ""
            return el

        _add("TotalLines", self.total_lines)
        _add("TotalNetAmount", self.total_net_amount)
        _add("TotalTaxableBasis", self.total_taxable_basis)
        _add("TotalTaxAmount", self.total_tax_amount)
        _add("TotalGrossAmount", self.total_gross_amount)

        tax_summary_el = SubElement(root, "Tax-Summary")
        for tax_line in self.tax_summary:
            tax_summary_el.append(tax_line.to_xml())

        return root
