from xml.etree.ElementTree import Element, SubElement
from .edi_invoice_delivery import InvoiceDelivery
from datetime import date
from enum import Enum

from .edi_line_item import InvoicePaymentMeans


class DocumentFunctionCode(Enum):
    """
    Possible values for <DocumentFunctionCode>

    Attribute:
        O: Original document
        D: Duplicate
        C: Correction
        R: Duplicate correction
    """

    O = "O"  # Oryginał
    D = "D"  # Duplikat
    C = "C"  # Korekta
    R = "R"  # Duplikat korekty


class InvoiceHeader:
    def __init__(
        self,
        invoice_number: str,
        invoice_date: date,
        sales_date: date,
        invoice_currency: str,
        invoice_payment_due_date: date,
        invoice_payment_terms: int,
        invoice_payment_means: InvoicePaymentMeans | None,
        document_function_code: DocumentFunctionCode,
        remarks: str,
        buyer_order_number: str | None,
        buyer_order_date: date | None,
        delivery: InvoiceDelivery,
    ):
        self.invoice_number = invoice_number
        self.invoice_date = invoice_date
        self.sales_date = sales_date
        self.invoice_currency = invoice_currency
        assert isinstance(
            invoice_payment_due_date, date
        ), "invoice_payment_due_date must be a date"
        self.invoice_payment_due_date = invoice_payment_due_date
        self.invoice_payment_terms = invoice_payment_terms
        self.invoice_payment_means = invoice_payment_means
        self.document_function_code = document_function_code
        self.remarks = remarks
        self.buyer_order_number = buyer_order_number
        if buyer_order_date is not None:
            assert isinstance(buyer_order_date, date), "buyer_order_date must be a date"
        self.buyer_order_date = buyer_order_date
        assert isinstance(
            delivery, InvoiceDelivery
        ), "delivery must be an InvoiceDelivery"
        self.delivery = delivery

    def _get_order_xml(self) -> Element:
        """Return XML Element <Order> containing the buyer order number and date."""
        order = Element("Order")
        order_number = SubElement(order, "BuyerOrderNumber")
        order_date = SubElement(order, "BuyerOrderDate")

        if self.buyer_order_number:
            order_number.text = str(self.buyer_order_number)

        if self.buyer_order_date:
            order_date.text = self.buyer_order_date.isoformat()

        return order

    def to_xml(self) -> Element:
        """Return XML Element <Header> containing invoice metadata and nested <Delivery>."""
        root = Element("Invoice-Header")

        def _add(tag, value):
            el = SubElement(root, tag)
            if isinstance(value, (date,)):
                el.text = value.isoformat()
            else:
                el.text = str(value).strip() if value is not None else ""
            return el

        _add("InvoiceNumber", self.invoice_number)
        _add("InvoiceDate", self.invoice_date)
        _add("SalesDate", self.sales_date)
        _add("InvoiceCurrency", self.invoice_currency)
        _add("InvoicePaymentDueDate", self.invoice_payment_due_date)
        _add("InvoicePaymentTerms", self.invoice_payment_terms)
        if self.invoice_payment_means is not None:
            _add(
                tag="InvoicePaymentMeans",
                value=self.invoice_payment_means.value,
            )
        _add("DocumentFunctionCode", self.document_function_code.value)
        _add("Remarks", self.remarks)

        # Add <Order/>
        root.append(self._get_order_xml())

        if self.delivery:
            root.append(self.delivery.to_xml())

        return root
