from xml.etree.ElementTree import Element, SubElement, tostring

from .edi_invoice_summary import InvoiceSummary
from .edi_buyer import Buyer
from .edi_seller import Seller
from .edi_invoice_header import InvoiceHeader
from .edi_line_item import LineItem


class InvoiceParties:
    def __init__(
        self,
        buyer: Buyer,
        seller: Seller,
    ):
        self.buyer = buyer
        self.seller = seller

    def to_xml(self) -> Element:
        """Return XML Element <Parties> containing <Buyer> and <Seller>."""
        root = Element("Invoice-Parties")
        if self.buyer:
            root.append(self.buyer.to_xml())
        if self.seller:
            root.append(self.seller.to_xml())
        return root


class Invoice:
    def __init__(
        self,
        header: InvoiceHeader,
        parties: InvoiceParties,
        lines: list[LineItem],
        summary: InvoiceSummary,
    ):
        self.header = header
        self.parties = parties
        self.lines = lines
        self.summary = summary

    def to_xml(self) -> Element:
        """Return the full XML <Invoice> structure with header, parties, line items, and summary."""
        root = Element("Document-Invoice")

        if self.header:
            root.append(self.header.to_xml())

        if self.parties:
            root.append(self.parties.to_xml())

        if self.lines:
            lines_el = SubElement(root, "Invoice-Lines")
            for line in self.lines:
                line_element = SubElement(lines_el, "Line")
                line_element.append(line.to_xml())

        # --- Summary ---
        if self.summary:
            root.append(self.summary.to_xml())

        return root

    def to_xml_string(self, encoding: str = "unicode") -> str:
        """Return XML as a string (for saving or sending)."""
        return tostring(self.to_xml(), encoding=encoding)
