from datetime import date

from app.domain.edi.models import ItemType
from app.domain.edi.models import UnitOfMeasure
from app.domain.edi.models import DocumentFunctionCode

from .exceptions import DocumentFunctionCodeError
from .exceptions import ItemTypeError
from .exceptions import UnitOfMeasureError
from .exceptions import InvoicePaymentMeansError

from ..edi.models.edi_line_item import InvoicePaymentMeans
from ..epp.models import DocumentHeader


def convert_unit_of_measure(base_unit: str) -> UnitOfMeasure:
    """
    Convert 'Item.base_unit' into the EDI 'Unit of Measure' field.
    According to the ISO 80000 standard with support for some units.
    """
    unit_of_measure = base_unit.strip().lower()
    if unit_of_measure == "kg":
        return UnitOfMeasure.KGM
    elif unit_of_measure == "opak" or unit_of_measure == "opak.":
        return UnitOfMeasure.PKG
    elif unit_of_measure == "szt" or unit_of_measure == "szt.":
        return UnitOfMeasure.PCE
    elif unit_of_measure == "m":
        return UnitOfMeasure.MTR
    else:
        raise UnitOfMeasureError(f"Unknown unit of measure: {unit_of_measure}")


def get_invoice_payment_terms(document_header: DocumentHeader) -> int | None:
    """
    Returns the number of days between the issue date and payment due date.
    Returns None if either date is missing or invalid.
    """
    issue_date = document_header.issue_date
    due_date = document_header.payment_due_date

    if not issue_date or not due_date:
        return 0

    assert isinstance(issue_date, date), "issue_date must be a date"
    assert isinstance(due_date, date), "payment_due_date must be a date"

    delta = (due_date - issue_date).days
    return max(delta, 0)


def get_invoice_payment_due_date(document_header: DocumentHeader) -> date:
    """
    Calculates and returns the payment due date for a given document. If the
    payment due date is specified, it returns that value. If not, it falls
    back to the document's issue date as the payment due date.
    """
    return document_header.payment_due_date or document_header.issue_date


def convert_to_item_type(i: int) -> ItemType:
    """
    Converts an integer value to its corresponding ItemType enumeration.

    This function maps integer values to predefined `ItemType` enumeration cases.
    If the provided integer does not map to a valid `ItemType`, an exception is
    raised.

    :raises ItemTypeError: If the provided integer value is not valid.
    """
    mapping = {
        1: ItemType.CU,
        2: ItemType.RC,
        3: ItemType.IN,
    }

    if mapping.get(i) is not None:
        return mapping[i]

    raise ItemTypeError(f"Invalid value cannot be converted to ItemType.")


def get_document_function_code(document_header: DocumentHeader) -> DocumentFunctionCode:
    """
    Convert 'Document.document_function_code' into the EDI 'Document Function Code' field.
    :raises: DocumentFunctionCodeError
    :returns: DocumentFunctionCode
    """
    if not isinstance(document_header.doc_type, str):
        raise DocumentFunctionCodeError(
            f"Document type from the header must be a string."
        )
    if document_header.doc_type.startswith("K"):
        return DocumentFunctionCode.C
    return DocumentFunctionCode.O


def get_invoice_payment_means(value: str) -> InvoicePaymentMeans:
    """
    Convert an EPP payment means to an invoice payment means.
    Throws an InvoicePaymentMeansError if the value is not supported.
    Usually value comes from the `DocumentHeader.payment_method_name` field.
    """
    value = value.strip().lower()

    if "przelew" in value:  # value.contains("przelew")
        return InvoicePaymentMeans.WIRE_TRANSFER
    else:
        raise InvoicePaymentMeansError(
            message=f'Unsupported payment method name ("{value}") cannot be converted into `InvoicePaymentMeans`',
            value=value,
        )
