from datetime import date

from loguru import logger

from app.domain.config import Config
from app.domain.edi.models import LineItem
from app.domain.edi.models import TaxCategoryCode
from app.domain.edi.models import InvoiceSummary
from app.domain.edi.models import InvoiceHeader
from app.domain.edi.models import Seller
from app.domain.edi.models import Buyer
from app.domain.edi.models import Invoice, InvoiceParties
from app.domain.edi.models import InvoiceDelivery
from app.domain.epp.models import Document, LineItem as EPPLineItem, DocumentHeader
from app.domain.epp import parse_date_or_null

from .build_tax_summary import build_tax_summary

from .exceptions import MissingCustomerNIP
from .exceptions import ILNError

from .utils import convert_to_item_type
from .utils import convert_unit_of_measure
from .utils import get_document_function_code
from .utils import get_invoice_payment_due_date
from .utils import get_invoice_payment_terms
from .utils import get_invoice_payment_means
from ..edi.models.edi_line_item import InvoicePaymentMeans


def convert_to_line_item(
    item: EPPLineItem,
    document: Document,
) -> LineItem:
    """
    Converts an EPP item to a (EDI) LineItem.

    Args:
        document: EPP document to read related data from.
        item: The EPP line item to convert.
    """
    product = document.get_item(item.item_code)
    name = item.one_time_service_description
    if not name:
        logger.warning(
            f"Item {item.item_code} does not have a one-time service description. Using product name instead."
        )
        name = product.name
    description = name.replace("\n", " ")
    assert (
        not "\n" in description
    ), "Line item description cannot contain newline characters"

    return LineItem(
        line_number=item.position_number,
        ean=str(product.barcode),
        item_type=convert_to_item_type(item.item_type),
        item_description=description,
        supplier_item_code=product.supplier_code,
        tax_rate=item.vat_rate_percent,
        net_amount=item.net_value,
        tax_amount=item.vat_value,
        unit_of_measure=convert_unit_of_measure(item.unit),
        invoice_unit_net_price=item.net_price,
        invoice_quantity=float(item.quantity_in_unit),
        tax_category_code=TaxCategoryCode.STANDARD,
    )


def convert_to_invoice_summary(
    document: Document, document_header: DocumentHeader
) -> InvoiceSummary:
    return InvoiceSummary(
        total_lines=document_header.number_of_items,
        total_net_amount=document_header.net_value,
        total_tax_amount=document_header.vat_value,
        total_gross_amount=document_header.gross_value,
        tax_summary_lines=build_tax_summary(document.line_items),
        total_taxable_basis=sum(item.net_value for item in document.line_items),
    )


def convert_to_delivery(
    document: Document, document_header: DocumentHeader, config: Config
) -> InvoiceDelivery:
    completion_date = document.get_delivery_completion_date(document_header)

    if not completion_date:
        raise ValueError(
            f"Document {document_header.get_full_number()} does not have a delivery completion date."
        )

    delivery_date = parse_date_or_null(completion_date.completion_date)

    return InvoiceDelivery(
        postal_code=document_header.customer_postal_code,
        street_and_number=document_header.customer_address,
        city_name=document_header.customer_city,
        delivery_date=delivery_date,
        name=document_header.customer_full_name,
        despatch_number=document_header.get_full_number(),
        delivery_location_number=config.get_delivery_location_iln(
            str(document_header.customer_nip)
        ),
    )


def convert_buyer(
    document: Document, document_header: DocumentHeader, config: Config
) -> Buyer:
    if not document_header.customer_nip:
        raise MissingCustomerNIP(
            f"Customer NIP is null or empty in document: {document_header.get_full_number()}.",
            document_number=document_header.get_full_number(),
        )
    assert isinstance(
        document_header.customer_nip, str
    ), "Customer NIP must be a string."
    customer_nip = str(document_header.customer_nip)  # Customer NIP can be null.
    contractor_id = document_header.customer_id
    contractor = document.get_contractor(contractor_id)

    if not contractor:
        raise ValueError(f"Contractor with ID: {contractor_id} not found.")

    iln = config.get_iln(customer_nip)

    if iln is None:
        raise ILNError(
            f"No ILN found for buyer with NIP: {customer_nip}. Please add it to the config.",
            nip=customer_nip,
        )

    if not isinstance(iln, str):
        raise ILNError(
            f"Buyer ILN must be a string, got instance of type {type(iln).__name__}",
            nip=customer_nip,
        )

    return Buyer(
        iln=iln,
        tax_id=document_header.customer_nip,
        name=document_header.customer_full_name,
        street_and_number=document_header.customer_address,
        city_name=document_header.customer_city,
        postal_code=document_header.customer_postal_code,
        country=contractor.ue_country_prefix or document_header.country_symbol,
    )


def _get_seller_account_number(
    document: Document, document_header: DocumentHeader
) -> str | None:
    customer_id = document_header.customer_id
    contractor = document.get_contractor(customer_id)
    return getattr(contractor, "bank_account_number", None)


def convert_seller(
    document: Document, config: Config, document_header: DocumentHeader
) -> Seller:
    sender_nip = str(document.file_header.sender_nip)
    iln = config.get_iln(sender_nip)

    if iln is None:
        raise ILNError(
            f"No ILN found for seller with NIP: {sender_nip}. Please add it to the config.",
            nip=sender_nip,
        )

    if not isinstance(iln, str):
        raise ILNError(
            f"Seller ILN must be a string, got instance of type {type(iln).__name__}"
        )

    _account_number: str | None = _get_seller_account_number(
        document=document, document_header=document_header
    )

    return Seller(
        iln=iln,
        tax_id=sender_nip,
        name=document.file_header.sender_long_name,
        city_name=document.file_header.sender_city,
        postal_code=document.file_header.sender_postal_code,
        account_number=_account_number,
        code_by_buyer=None,
        court_and_capital_information=None,
        street_and_number=document.file_header.sender_address,
        country=document.file_header.eu_country_prefix,
    )


def get_buyer_order_date(header: DocumentHeader) -> date:
    v = header.order
    if isinstance(v, str):
        v = parse_date_or_null(v)
    return v


def _get_invoice_payment_means(header: DocumentHeader) -> InvoicePaymentMeans | None:
    _raw_payment_means = header.payment_method_name
    if not _raw_payment_means:
        logger.warning(
            f"Payment means is not found for document: {header.get_full_number()}. Using null."
        )
        return None
    return get_invoice_payment_means(_raw_payment_means)


def convert_header(
    document: Document, header: DocumentHeader, config: Config
) -> InvoiceHeader:
    invoice_payment_terms = get_invoice_payment_terms(header)
    invoice_payment_due_date = get_invoice_payment_due_date(header)
    invoice_payment_means = _get_invoice_payment_means(header)
    # Document format does not provide the date of the order.
    # ---------------------------------------------------
    buyer_order_date = None

    return InvoiceHeader(
        invoice_number=header.get_full_number(),
        invoice_date=header.issue_date,
        sales_date=header.sale_date,
        invoice_currency=header.currency,
        document_function_code=get_document_function_code(header),
        remarks=header.remarks,
        buyer_order_number=header.order_number,
        invoice_payment_due_date=invoice_payment_due_date,
        invoice_payment_means=invoice_payment_means,
        invoice_payment_terms=invoice_payment_terms,
        buyer_order_date=buyer_order_date,
        delivery=convert_to_delivery(
            document=document, document_header=header, config=config
        ),
    )


def convert_document(document: Document, config: Config):
    """
    :raises ILNException: Invalid ILN
    """
    result: list[Invoice] = []

    for document_header in document.document_headers:
        parties = InvoiceParties(
            buyer=convert_buyer(document, document_header, config),
            seller=convert_seller(document, config, document_header=document_header),
        )
        invoice = Invoice(
            header=convert_header(document, header=document_header, config=config),
            parties=parties,
            lines=[
                convert_to_line_item(line, document) for line in document.line_items
            ],
            summary=convert_to_invoice_summary(
                document, document_header=document_header
            ),
        )
        result.append(invoice)

    return result
