from collections import defaultdict
from typing import List

from app.domain.edi.models import TaxSummaryLine
from app.domain.epp.models import LineItem


def build_tax_summary(lines: List[LineItem]) -> List[TaxSummaryLine]:
    """
    Aggregate line items by VAT rate into the TaxSummaryLine list.
    Assumes numeric fields are already properly typed.
    """
    buckets = defaultdict(lambda: {"net": 0.0, "vat": 0.0, "gross": 0.0})

    for li in lines:
        rate = li.vat_rate_percent
        buckets[rate]["net"] += li.net_value
        buckets[rate]["vat"] += li.vat_value
        buckets[rate]["gross"] += li.gross_value

    result = []
    for rate, sums in buckets.items():
        result.append(
            TaxSummaryLine(
                tax_rate=rate,
                tax_category_code="S",  # TODO: tax_category_code=S
                tax_amount=round(sums["vat"], 2),
                taxable_basic=round(sums["net"], 2),
                taxable_amount=round(sums["net"], 2),
                gross_amount=round(sums["gross"], 2),
            )
        )
    return result
