from pathlib import Path

from loguru import logger


class InvalidConfigError(Exception):
    """Raised when the configuration is invalid."""

    ...


class Config:
    """Represents the configuration of the application."""

    def __init__(
        self,
        input_dir: Path = None,
        output_dir: Path = None,
        processed_dir: Path = None,
        gln=None,
        ilns=None,
        delivery_location_ilns=None,
    ):
        self.input_dir = input_dir
        self.output_dir = output_dir
        self.processed_dir = processed_dir
        self.gln = gln
        self.ilns = ilns
        self.delivery_location_ilns = delivery_location_ilns

    @staticmethod
    def from_dict(config_dict: dict) -> "Config":
        from os import getcwd
        from os.path import join

        current_dir = getcwd()

        if "GLN" in config_dict:
            logger.warning("GLN is deprecated. Use ILNs instead.")

        return Config(
            input_dir=Path(join(current_dir, config_dict["EPP_INPUT_DIR"])),
            output_dir=Path(join(current_dir, config_dict["EDI_OUTPUT_DIR"])),
            processed_dir=Path(join(current_dir, config_dict["PROCESSED_DIR"])),
            gln=config_dict["GLN"],
            ilns=config_dict["ILNs"],
            delivery_location_ilns=config_dict["DELIVERY_LOCATION_ILNs"],
        )

    def get_iln(self, nip: str) -> str | None:
        """Get ILN from the ILNs dictionary."""
        assert isinstance(nip, str), "NIP must be a string."
        return self.ilns.get(nip, None)

    def get_delivery_location_iln(self, nip: str) -> str | None:
        """Get ILN from the delivery_location_ilns dictionary."""
        assert isinstance(nip, str), "NIP must be a string."
        return self.delivery_location_ilns.get(nip, None)

    # noinspection PyMethodMayBeStatic
    def is_valid(
        self,
        check_paths=True,
        throws=False,
    ) -> bool:
        """
        Check if the configuration is valid.
        :param throws: If True, throws an InvalidConfigError if the configuration is invalid.
        :param check_paths: If True, checks if the paths exist.
        :raises InvalidConfigError: If the configuration is invalid and throws is True.
        :return: True if the configuration is valid, False otherwise.
        """
        try:
            assert isinstance(self.input_dir, Path), "input_dir must be a Path object."
            assert isinstance(
                self.output_dir, Path
            ), "output_dir must be a Path object."
            assert isinstance(
                self.processed_dir, Path
            ), "output_dir must be a Path object."

            if check_paths:
                assert self.input_dir.exists(), "input_dir does not exist."
                assert self.output_dir.exists(), "output_dir does not exist."
                assert self.processed_dir.exists(), "processed_dir does not exist."
            return True

        except AssertionError as e:
            if throws:
                raise InvalidConfigError(str(e)) from e
            return False
