from rest_framework.viewsets import ModelViewSet
from rest_framework.decorators import action
from rest_framework.response import Response

from django_filters.rest_framework import DjangoFilterBackend

from django.utils.module_loading import import_string
from django.conf import settings

from .backends import SimpleOrderingBackend
from .pagination import BasePagination

def get_permission_classes():
    """
    Retrieve and dynamically import permission classes from settings.
    :return: List
    """
    permission_classes = getattr(settings, 'USER_GUIDE_PERMISSION_CLASSES', [])
    return [import_string(permission_class) for permission_class in permission_classes]

def get_authentication_classes():
    """
    Retrieve and dynamically import authentication classes from settings.
    :return: List
    """
    authentication_classes = getattr(settings, 'USER_GUIDE_AUTHENTICATION_CLASSES', [])
    return [import_string(auth_class) for auth_class in authentication_classes]


class BaseViewSet(ModelViewSet):
    """
    A custom base view set for handling serializers dynamically based on the action.

    This class extends the ModelViewSet and provides advanced functionality for
    serializers. It allows the developer to define different serializers for
    different actions like list, create, retrieve, and update. This is useful when
    different serializers are needed for different endpoints of the same resource.
    """

    update_serializer = None
    create_serializer = None
    read_serializer = None
    list_serializer = None
    all_serializer = None

    filter_backends = [
        SimpleOrderingBackend,
        DjangoFilterBackend,
    ]

    pagination_class = BasePagination

    permission_classes = get_permission_classes()
    authentication_classes = get_authentication_classes()

    # noinspection PyUnusedLocal
    @action(detail=False, methods=['GET'], url_path='all')
    def all(self, request, *args, **kwargs):
        """
        Returns all objects of the resource.
        """
        queryset = self.filter_queryset(self.get_queryset())
        serializer = self.get_serializer(queryset, many=True)
        return Response(serializer.data)

    def get_serializer_mapping(self):
        """
        Returns dictionary with keys of action names and serializer_classes in values.
        To use in get_serializer_class() to get the appropriate class to request.
        """
        return {
            'list': self.list_serializer,
            'create': self.create_serializer,
            'retrieve': self.read_serializer,
            'update': self.update_serializer,
            'all': self.all_serializer,
            'metadata': self.list_serializer or self.read_serializer
        }

    def get_serializer_context(self):
        """
        Get default serializer context.

        Returns:
            { user: self.request.user, **super().get_serializer_context() }
        """
        return {
            **super().get_serializer_context(),
            'user': self.request.user,
        }

    def get_serializer_class(self):
        """
        Returns the appropriate serializer class.
        It's using get_serializers_mapping() from BaseViewSet to match action to serializer_class.
        Raises MethodNotAllowed, if no serializer is set for action and self.serializer class is not set.
        """
        serializer_class = self.get_serializer_mapping().get(
            self.action,
            None
        )

        if serializer_class is None:
            print(">>> No serializer found for action: ", self.action, " <<<")
            return self.list_serializer or self.read_serializer
        return serializer_class